# ideas/permissions.py

from rest_framework import permissions
from rest_framework.permissions import BasePermission, SAFE_METHODS

class IsAdmin(permissions.BasePermission):
    """
    دسترسی سفارشی برای اینکه فقط کاربران ادمین (is_staff) مجاز باشند.
    """
    def has_permission(self, request, view):
        # کاربر باید لاگین کرده باشد و یک کاربر staff (ادمین) باشد.
        return request.user and request.user.is_staff

class IsOwnerOrAdmin(permissions.BasePermission):
    """
    دسترسی سفارشی برای اینکه فقط مالک شیء یا ادمین بتواند آن را ویرایش کند.
    """
    def has_object_permission(self, request, view, obj):
        # دسترسی خواندن برای همه مجاز است.
        if request.method in permissions.SAFE_METHODS:
            return True

        # دسترسی نوشتن فقط برای مالک یا ادمین مجاز است.
        # ما چک می‌کنیم که شیء فیلد 'author' را داشته باشد.
        if hasattr(obj, 'author'):
            return obj.author == request.user or request.user.is_staff

        # برای مدل‌هایی مثل Attachment که شاید author نداشته باشند،
        # به نویسنده ایده اصلی دسترسی می‌دهیم.
        if hasattr(obj, 'idea') and hasattr(obj.idea, 'author'):
            return obj.idea.author == request.user or request.user.is_staff

        # اگر هیچکدام از شرایط بالا برقرار نبود، اجازه دسترسی نده.
        return False
    
# backend/ideas/permissions.py
from rest_framework.permissions import BasePermission

class IsReviewer(BasePermission):
    """
    اجازه دسترسی فقط به کاربرانی که نقش Reviewer دارند.
    """
    def has_permission(self, request, view):
        return bool(request.user and request.user.is_authenticated and request.user.role == 'reviewer')
    
class IsReviewerOrAdmin(BasePermission):
    """
    Allows access only to users with role 'reviewer' or 'admin'
    """

    def has_permission(self, request, view):
        user = request.user
        if not user or not user.is_authenticated:
            return False

        return user.role in ["reviewer", "admin"]
    
class IsReviewerOrAdminOrReadOnly(BasePermission):
    def has_permission(self, request, view):
        # Read-only برای همه
        if request.method in SAFE_METHODS:
            return True

        # Write فقط برای reviewer یا admin
        user = request.user
        if not user.is_authenticated:
            return False

        return (
            user.is_staff or
            user.groups.filter(name="reviewer").exists()
        )
from rest_framework.permissions import BasePermission


class CanEvaluateIdea(BasePermission):
    def has_permission(self, request, view):
        return (
            request.user
            and request.user.is_authenticated
            and request.user.role in ["admin", "reviewer"]
        )
class CanEditEvaluation(BasePermission):
    """
    فقط ادمین یا ریویوئر اجازه ویرایش ارزیابی خودش را دارد
    """

    def has_permission(self, request, view):
        return (
            request.user
            and request.user.is_authenticated
            and (
                request.user.is_staff
                or request.user.groups.filter(name="reviewer").exists()
            )
        )
class IsAdminOnly(BasePermission):
    def has_permission(self, request, view):
        return request.user.is_staff