# backend/ideas/views.py
from rest_framework.filters import OrderingFilter
import csv
from django.http import HttpResponse
from django.utils import timezone
from django.shortcuts import get_object_or_404
from django_filters.rest_framework import DjangoFilterBackend
import django_filters

from ideas.constants.notification_types import NotificationType
from ideas.services.decision import calculate_decision
from ideas.services.notifications import create_notification
from .models import  Idea, RefinementAnswer, RefinementQuestion, RefinementReview
from rest_framework import viewsets, permissions, status, filters
from rest_framework.decorators import action, api_view, permission_classes
from rest_framework.response import Response
from rest_framework.views import APIView
from rest_framework.permissions import IsAuthenticated,IsAdminUser
from .models import (
    ActivityLog, User, Tag, Campaign, Category, Idea, Comment,EvaluationCriterion, Attachment, Notification,RefinementQuestion,RefinementReview, RefinementAnswer,
)
from .models import EvaluationResult

from rest_framework.generics import ListAPIView
from rest_framework import generics
from .permissions import CanEditEvaluation, CanEvaluateIdea, IsReviewer,IsReviewerOrAdmin,IsReviewerOrAdminOrReadOnly
from ideas.services.scoring import calculate_final_score

# اگر فایل permissions.py دارید از آنجا بخوانید، اگر نه کلاس‌ها پایین تعریف شده‌اند
from .permissions import IsAdmin, IsOwnerOrAdmin 
from .serializers import (
    ActivityLogSerializer, EvaluateIdeaSerializer, RefinementQuestionSerializer, RefinementReviewCreateSerializer, RefinementReviewDisplaySerializer, UserSerializer, TagSerializer, CampaignSerializer, CategorySerializer,
    IdeaListSerializer, IdeaDetailSerializer, CommentSerializer, AttachmentSerializer,
    NotificationSerializer,ArchiveIdeaSerializer,RefinementReviewSerializer,EvaluationCriterionSerializer,
)
from .pagination import NotificationPagination, StandardResultsSetPagination
from rest_framework.exceptions import PermissionDenied

# ---------- Permissions Definition ----------
# (اگر این‌ها در فایل permissions.py هستند، می‌توانید این بخش را حذف کنید)
class RefinementQuestionListView(APIView):
    permission_classes = [IsAuthenticated]

    def get(self, request):
        qs = RefinementQuestion.objects.filter(is_active=True).order_by('order')
        ser = RefinementQuestionSerializer(qs, many=True)
        return Response(ser.data)


class RefinementReviewView(APIView):
    permission_classes = [IsReviewerOrAdminOrReadOnly]
    def get_permissions(self):
        if self.request.method == "GET":
            return [IsAuthenticated()]
        return [IsReviewerOrAdmin()]
    def get(self, request, idea_id):
        review = RefinementReview.objects.filter(idea_id=idea_id).first()
        if not review:
            return Response(None, status=status.HTTP_200_OK)
        serializer = RefinementReviewDisplaySerializer(review)
        return Response(serializer.data)
    def post(self, request, idea_id):
        idea = get_object_or_404(Idea, id=idea_id)
        serializer = RefinementReviewCreateSerializer(
            data=request.data,
            context={
                "request": request,
                "idea": idea,   # ✅ این کانتکست برای سریالایزر ضروری است
            }
        )
        serializer.is_valid(raise_exception=True)
        review = serializer.save()
    
        # اگر می‌خواهید خروجی با فرمت نمایش باشد، از سریالایزر نمایش استفاده کنید
        # اما اول مطمئن شوید RefinementReviewSerializer در فایل views.py ایمپورت شده است
        return Response(
            RefinementReviewDisplaySerializer(review).data,
            status=status.HTTP_201_CREATED
        )
#Evaluation
class EvaluationCriteriaListView(ListAPIView):
    queryset = EvaluationCriterion.objects.all()
    serializer_class = EvaluationCriterionSerializer
    permission_classes = [IsAuthenticated]



#--------------------------------------------------------------------------------------------------------
class IsAuthorOrAdmin(permissions.BasePermission):
    """
    فقط نویسنده‌ی ایده یا ادمین می‌تونن ویرایش کنن.
    """
    def has_object_permission(self, request, view, obj):
        if request.method in permissions.SAFE_METHODS:
            return True
        return obj.author == request.user or request.user.role == "admin"

class IsReviewer(permissions.BasePermission): # <<< کلاس جدید برای داورها
    """
    فقط کاربرانی که نقش reviewer دارند دسترسی دارند.
    """
    def has_permission(self, request, view):
        return bool(request.user and request.user.is_authenticated and request.user.role == 'reviewer')


# ---------- Users / Categories ----------
class UserViewSet(viewsets.ModelViewSet):
    queryset = User.objects.all().order_by('id')
    serializer_class = UserSerializer
    permission_classes = [IsAdmin]

class CategoryViewSet(viewsets.ModelViewSet):
    queryset = Category.objects.all().order_by('name')
    serializer_class = CategorySerializer
    permission_classes = [IsAdmin]

# ---------- Common mixin for not-deleted ----------
class NotDeletedQuerysetMixin:
    def get_queryset(self):
        qs = super().get_queryset()
        # برای دیدن همه در این اکشن‌ها
        if getattr(self, 'action', None) in ('trash', 'restore', 'hard_delete'):
            return qs.model.objects.all()
        # امکان override با ?include_deleted=1
        if getattr(self, 'request', None) and self.request.query_params.get('include_deleted') == '1':
            return qs.model.objects.all()
        return qs.filter(is_deleted=False)


# ---------- Campaigns ----------
class CampaignViewSet(NotDeletedQuerysetMixin, viewsets.ModelViewSet):
    queryset = Campaign.objects.all().order_by('-start_date')
    serializer_class = CampaignSerializer
    filter_backends = [DjangoFilterBackend, filters.SearchFilter, filters.OrderingFilter]
    pagination_class = StandardResultsSetPagination
    search_fields = ['title', 'description']
    ordering_fields = [
        'id',
        'created_at',
        'archived_at',
        'stage',
        'campaign__title',
        'score',
        'last_stage_change',
    ]
    def get_permissions(self):
        if self.action in ['create', 'update', 'partial_update', 'destroy', 'restore', 'hard_delete', 'trash']:
            self.permission_classes = [IsAdmin]
        else:
            self.permission_classes = [permissions.IsAuthenticated]
        return super().get_permissions()

    def destroy(self, request, *args, **kwargs):
        obj = self.get_object()
        obj.soft_delete(request.user)
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=True, methods=['post'], permission_classes=[permissions.IsAdminUser])
    def restore(self, request, pk=None):
        obj = Campaign.objects.filter(pk=pk).first()
        if not obj:
            return Response({'detail': 'Not found.'}, status=status.HTTP_404_NOT_FOUND)
        obj.restore()
        ser = self.get_serializer(obj, context={'request': request})
        return Response(ser.data)

    @action(detail=True, methods=['delete'], url_path='hard-delete', permission_classes=[permissions.IsAdminUser])
    def hard_delete(self, request, pk=None):
        obj = Campaign.objects.filter(pk=pk).first()
        if not obj:
            return Response({'detail': 'Not found.'}, status=status.HTTP_404_NOT_FOUND)
        obj.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=False, methods=['get'], permission_classes=[permissions.IsAdminUser])
    def trash(self, request):
        qs = Campaign.objects.filter(is_deleted=True).order_by('-start_date')
        page = self.paginate_queryset(qs)
        ser = self.get_serializer(page or qs, many=True)
        return self.get_paginated_response(ser.data) if page is not None else Response(ser.data)

    @action(detail=True, methods=['get'], permission_classes=[IsAdmin])
    def export_ideas(self, request, pk=None):
        campaign = self.get_object()
        response = HttpResponse(content_type='text/csv')
        response['Content-Disposition'] = f'attachment; filename="ideas_campaign_{campaign.pk}.csv"'
        writer = csv.writer(response)
        writer.writerow(['ID', 'Title', 'Author', 'Stage', 'Net Score', 'TRL', 'Submitted At'])
        for idea in campaign.ideas.filter(is_deleted=False).select_related('author'):
            writer.writerow([
                idea.pk, idea.title, idea.author.username, idea.get_stage_display(),
                idea.score, idea.trl or '',
                idea.created_at.strftime('%Y-%m-%d %H:%M') if idea.created_at else ''
            ])
        return response

class IdeaFilter(django_filters.FilterSet):
    campaign__category = django_filters.NumberFilter(field_name='campaign__category')

    class Meta:
        model = Idea
        fields = ['stage', 'campaign', 'author', 'campaign__category']
# ---------- Ideas ----------
class IdeaViewSet(NotDeletedQuerysetMixin, viewsets.ModelViewSet):
    queryset = (
        Idea.objects
        .all()
        .prefetch_related('likes', 'dislikes', 'comments', 'tags')
        .select_related('author', 'campaign')
        .order_by('-id')
    )
    pagination_class = NotificationPagination
    filter_backends = [DjangoFilterBackend, filters.SearchFilter, filters.OrderingFilter,]
    filterset_class = IdeaFilter
    search_fields = ['title', 'description']
    ordering_fields = [
        'id',
        'created_at',
        'archived_at',
        'stage',
        'campaign__title',
        'score',
        'last_stage_change',
    ]    
    permission_classes = [IsAuthorOrAdmin]

    def get_serializer_class(self):
        # ✅ Archive Page (List Only)
        if self.action == 'list' and self.request.query_params.get('is_archived') == 'true':
            return ArchiveIdeaSerializer
        if self.action == 'list':
            return IdeaListSerializer
        elif self.action == 'retrieve':
            return IdeaDetailSerializer
        elif self.action in ['update', 'partial_update']:
            return IdeaDetailSerializer
        return IdeaDetailSerializer
        

    # متد کمکی برای چک کردن آرشیو
    def ensure_not_archived(self, idea):
        if idea.is_archived:
            raise PermissionDenied("Archived ideas are read-only.")

    # اصلاح get_queryset: ترکیب منطق حذف نشده‌ها + فیلتر آرشیو
    def get_queryset(self):
        # ابتدا منطق mixin را صدا می‌زنیم (فیلتر is_deleted=False)
        qs = super().get_queryset()
        
        # سپس فیلتر آرشیو را اعمال می‌کنیم
        archived = self.request.query_params.get('is_archived')
        if archived is not None:
            if archived.lower() == 'true':
                qs = qs.filter(is_archived=True)
            elif archived.lower() == 'false':
                qs = qs.filter(is_archived=False)
        return qs

    def get_permissions(self):
        if self.action == 'review':
             self.permission_classes = [IsReviewer]
        elif self.action in ['destroy', 'restore', 'hard_delete', 'trash', 'change_stage', 'archive']:
            self.permission_classes = [IsAdmin]
        elif self.action in ['update', 'partial_update']:
            self.permission_classes = [IsOwnerOrAdmin]
        else:
            self.permission_classes = [permissions.IsAuthenticated]
        return super().get_permissions()

    def get_serializer_context(self):
        return {'request': self.request}

    def perform_create(self, serializer):
        serializer.save(author=self.request.user)

    def destroy(self, request, *args, **kwargs):
        obj = self.get_object()
        obj.soft_delete(request.user)
        return Response(status=status.HTTP_204_NO_CONTENT)

    # اصلاح perform_update و حذف کدهای تکراری
    def perform_update(self, serializer):
        idea = serializer.save()

        # بروزرسانی متادیتا
        idea.edited = True
        idea.edited_at = timezone.now()
        idea.edited_by = self.request.user
        idea.save(update_fields=['edited', 'edited_at', 'edited_by'])

        # مدیریت آپلود فایل (یک بار کافیست)
        file_obj = self.request.FILES.get('attachment')
        if file_obj:
            file_type = ''
            if hasattr(file_obj, 'name') and '.' in file_obj.name:
                file_type = file_obj.name.split('.')[-1].lower()
            
            Attachment.objects.create(
                idea=idea,
                file=file_obj,
                file_type=file_type
            )

    def partial_update(self, request, *args, **kwargs):
        serializer = self.get_serializer(self.get_object(), data=request.data, partial=True)
        serializer.is_valid(raise_exception=True)
        self.perform_update(serializer)
        return Response(serializer.data)

    # ------------------ Actions ------------------

    @action(detail=True, methods=['post']) 
    def review(self, request, pk=None):
        idea = self.get_object()
        
        text = request.data.get('text', '').strip()
        if not text:
            return Response({'detail': 'Review text is required.'}, status=status.HTTP_400_BAD_REQUEST)
        
        comment = Comment.objects.create(
            idea=idea,
            author=request.user,
            text=text
        )

        if idea.author != request.user:
            create_notification(
                recipient=idea.author,
                actor=self.request.user,
                verb=NotificationType.COMMENT,
                target=idea
            )
        
        return Response(CommentSerializer(comment).data, status=status.HTTP_201_CREATED)

    @action(detail=True, methods=['get'], url_path='reviews')
    def get_reviews(self, request, pk=None):
        idea = self.get_object()
        comments = Comment.objects.filter(idea=idea).order_by('-created_at')
        return Response(CommentSerializer(comments, many=True).data)

    @action(detail=True, methods=['post'], permission_classes=[IsAdminUser])
    def restore(self, request, pk=None):
        # اینجا باید چک کنید منظورتان restore از سطل زباله است یا از آرشیو؟
        # کد شما currently فیلدهای آرشیو را ریست می‌کند.
        idea = self.get_object()
        if idea.is_archived:
            idea.is_archived = False
            idea.archived_at = None
            idea.archived_by = None
            idea.save()
            return Response({"status": "restored from archive"})
        
        # اگر منظورتان soft-delete restore است:
        if idea.is_deleted:
            idea.restore() # فرض بر این است متد restore در مدل دارید
            return Response({"status": "restored from trash"})

        return Response({"status": "nothing to restore"})

    @action(detail=True, methods=['delete'], url_path='hard-delete', permission_classes=[permissions.IsAdminUser])
    def hard_delete(self, request, pk=None):
        # برای hard delete باید از objects.all() استفاده کنید چون get_object معمولی فیلتر شده است
        obj = Idea.objects.filter(pk=pk).first()
        if not obj:
            return Response({'detail': 'Not found.'}, status=404)
        obj.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=False, methods=['get'], permission_classes=[permissions.IsAdminUser])
    def trash(self, request):
        qs = Idea.objects.filter(is_deleted=True).order_by('-id')
        page = self.paginate_queryset(qs)
        ser = IdeaListSerializer(page or qs, many=True, context={'request': request})
        return self.get_paginated_response(ser.data) if page is not None else Response(ser.data)

    @action(detail=True, methods=['post'])
    def change_stage(self, request, pk=None):
        idea = self.get_object()
        new_stage = request.data.get('new_stage')

        if not new_stage or new_stage not in dict(Idea.STAGE_CHOICES):
            return Response(
                {"detail": "Invalid or missing 'new_stage'"},
                status=status.HTTP_400_BAD_REQUEST
            )

        old_stage = idea.stage
        idea.stage = new_stage
        idea.last_stage_change = timezone.now()
        idea.save(update_fields=['stage', 'last_stage_change'])
        if old_stage != idea.stage:
            create_notification(
                recipient=idea.author,
                actor=request.user,
                verb=NotificationType.STAGE_CHANGE,
                target=idea
            )
        ActivityLog.objects.create(
            idea=idea,
            previous_stage=old_stage,
            new_stage=new_stage,
            notes=request.data.get("notes", ""),
            changed_by=request.user
        )

        return Response(IdeaDetailSerializer(idea, context=self.get_serializer_context()).data)

    @action(detail=True, methods=['post'])
    def like(self, request, pk=None):
        idea = self.get_object()
        self.ensure_not_archived(idea) # استفاده از self
        user = request.user
        if idea.likes.filter(id=user.id).exists():
            idea.likes.remove(user)
        else:
            idea.likes.add(user)
            idea.dislikes.remove(user)
        idea.score = idea.likes.count() - idea.dislikes.count()
        idea.save(update_fields=['score'])
        return Response(IdeaDetailSerializer(idea, context=self.get_serializer_context()).data)

    @action(detail=True, methods=['post'])
    def dislike(self, request, pk=None):
        idea = self.get_object()
        self.ensure_not_archived(idea) # استفاده از self
        user = request.user
        if idea.dislikes.filter(id=user.id).exists():
            idea.dislikes.remove(user)
        else:
            idea.dislikes.add(user)
            idea.likes.remove(user)
        idea.score = idea.likes.count() - idea.dislikes.count()
        idea.save(update_fields=['score'])
        return Response(IdeaDetailSerializer(idea, context=self.get_serializer_context()).data)
    
    @action(detail=True, methods=['get'])
    def engagement(self, request, pk=None):
        obj = self.get_object()
        return Response({
            'liked_by': list(obj.likes.values_list('username', flat=True)),
            'disliked_by': list(obj.dislikes.values_list('username', flat=True)),
        })

    @action(
        detail=True,
        methods=["post"],
        permission_classes=[IsAdminUser]
    )
    def archive(self, request, pk=None):
        idea = self.get_object()

        if idea.is_archived:
            return Response(
                {"detail": "Idea already archived"},
                status=status.HTTP_400_BAD_REQUEST
            )

        idea.is_archived = True
        idea.archived_at = timezone.now()
        idea.archived_by = request.user
        idea.save()

        return Response({"status": "archived"})
    

    @action(
        detail=True,
        methods=["post", "patch"],
        url_path="evaluate",
        permission_classes=[IsAuthenticated],
    )
    def evaluate(self, request, pk=None):
        idea = self.get_object()
        user = request.user

        # ---------- PATCH = edit evaluation ----------
        if request.method == "PATCH":
            evaluation = EvaluationResult.objects.filter(
                idea=idea,
                evaluator=user
            ).first()

            if not evaluation:
                return Response(
                    {"detail": "You have not evaluated this idea."},
                    status=status.HTTP_400_BAD_REQUEST
                )

            scores = request.data.get("scores")
            if not isinstance(scores, dict):
                return Response(
                    {"detail": "Invalid scores format."},
                    status=status.HTTP_400_BAD_REQUEST
                )

            evaluation.scores = scores
            evaluation.save()

            final_score = calculate_final_score(evaluation.scores)

            h_level = idea.h_level   # ✅ اینجا اصلاح شد

            decision = calculate_decision(
                final_score=final_score,
                h_level=h_level
            )

            evaluation.final_score = final_score
            evaluation.save(update_fields=["final_score"])

            idea.final_score = final_score
            idea.save(update_fields=["final_score"])

            return Response({
                "final_score": final_score,
                "scores": evaluation.scores,
                "decision": decision,          # ✅ حالا تعریف شده
                "current_stage": idea.stage,   # بدون side‑effect
                "h_level": h_level,
            })

        # ---------- POST = create evaluation ----------
        serializer = EvaluateIdeaSerializer(
            data=request.data,
            context={
                "idea": idea,
                "request": request,
            }
        )
        serializer.is_valid(raise_exception=True)
        evaluation = serializer.save()

        decision = calculate_decision(
            final_score=evaluation.final_score,
            h_level=evaluation.h_level
        )

        return Response(
            {
                "final_score": evaluation.final_score,
                "h_level": evaluation.h_level,
                "scores": evaluation.scores,
                "decision": decision,   # ✅ برای فرانت
            },
            status=status.HTTP_201_CREATED
        )



class ActivityLogViewSet(viewsets.ModelViewSet):
    queryset = ActivityLog.objects.all()
    serializer_class = ActivityLogSerializer
    permission_classes = [IsAuthenticated]

    def get_queryset(self):
        idea_id = self.kwargs.get("idea_id")
        return ActivityLog.objects.filter(idea_id=idea_id).order_by("-changed_at")

    def perform_create(self, serializer):
        idea_id = self.kwargs.get("idea_id")
        serializer.save(idea_id=idea_id, changed_by=self.request.user)


# ---------- Comments (nested under ideas) ----------
class CommentViewSet(viewsets.ModelViewSet):
    serializer_class = CommentSerializer
    pagination_class = NotificationPagination

    def get_queryset(self):
        idea_id = self.kwargs['idea_pk']
        return Comment.objects.filter(idea_id=idea_id, is_deleted=False).order_by('-created_at')

    def get_permissions(self):
        if self.action in ['destroy', 'restore', 'hard_delete', 'trash']:
            self.permission_classes = [IsAdmin]
        elif self.action in ['update', 'partial_update']:
            self.permission_classes = [IsOwnerOrAdmin]
        else:
            self.permission_classes = [permissions.IsAuthenticated]
        return super().get_permissions()

    def perform_create(self, serializer):
        idea = Idea.objects.get(pk=self.kwargs['idea_pk'])
        comment = serializer.save(author=self.request.user, idea=idea)
        if idea.author != self.request.user:
            create_notification(
                recipient=idea.author,
                actor=self.request.user,
                verb=NotificationType.COMMENT,
                target=idea
            )

    # def perform_update(self, serializer):
    #     instance = serializer.save()
    #     instance.edited = True
    #     instance.edited_at = timezone.now()
    #     instance.edited_by = self.request.user
    #     instance.save(update_fields=['edited', 'edited_at', 'edited_by'])

    def destroy(self, request, *args, **kwargs):
        obj = self.get_object()
        obj.soft_delete(request.user)
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=True, methods=['post'])
    def restore(self, request, idea_pk=None, pk=None):
        obj = self.get_object()
        obj.restore()
        return Response(self.get_serializer(obj).data)

    @action(detail=True, methods=['delete'], url_path='hard-delete')
    def hard_delete(self, request, idea_pk=None, pk=None):
        obj = self.get_object()
        obj.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=False, methods=['get'])
    def trash(self, request, idea_pk=None):
        qs = Comment.objects.filter(idea_id=idea_pk, is_deleted=True).order_by('-deleted_at')
        page = self.paginate_queryset(qs)
        ser = self.get_serializer(page or qs, many=True)
        return self.get_paginated_response(ser.data) if page is not None else Response(ser.data)


# ---------- Attachments (nested) ----------
class AttachmentViewSet(viewsets.ModelViewSet):
    serializer_class = AttachmentSerializer

    def get_queryset(self):
        return Attachment.objects.filter(idea_id=self.kwargs['idea_pk'])

    def get_permissions(self):
        if self.action == 'destroy':
            self.permission_classes = [IsOwnerOrAdmin]
        else:
            self.permission_classes = [permissions.IsAuthenticated]
        return super().get_permissions()

    def perform_create(self, serializer):
        idea = Idea.objects.get(pk=self.kwargs['idea_pk'])
        file_obj = self.request.data.get('file')
        file_type = ''
        if file_obj and hasattr(file_obj, 'name') and '.' in file_obj.name:
            file_type = file_obj.name.split('.')[-1].lower()
        serializer.save(idea=idea, file_type=file_type)


# ---------- Analytics ----------
class AnalyticsSummaryAPIView(APIView):
    permission_classes = [permissions.IsAuthenticated]

    def get(self, request, format=None):
        total_ideas = Idea.objects.filter(is_deleted=False).count()
        active_campaigns = Campaign.objects.filter(is_deleted=False, status='active').count()
        implemented_ideas = Idea.objects.filter(stage='portfolio').count()
        satisfaction_rate = (implemented_ideas / total_ideas * 100) if total_ideas > 0 else 0
        funnel_stats = {
            display: Idea.objects.filter(is_deleted=False, stage=key).count()
            for key, display in Idea.STAGE_CHOICES
        }
        return Response({
            'total_ideas': total_ideas,
            'active_campaigns': active_campaigns,
            'satisfaction_rate': round(satisfaction_rate, 2),
            'funnel_pipeline': funnel_stats
        })


# ---------- Notifications ----------
class NotificationViewSet(viewsets.ReadOnlyModelViewSet):
    serializer_class = NotificationSerializer
    permission_classes = [IsAuthenticated]

    def get_queryset(self):
        return (
            Notification.objects
            .filter(recipient=self.request.user)
            .select_related('actor', 'target_content_type')
            .order_by('-timestamp')
        )
        print("🔥 NOTIF COUNT:", qs.count())
        for n in qs:
            print("🧪 notif:", n.id, n.verb, n.unread)

        return qs
    
    @action(detail=True, methods=["post"], url_path="mark-read")
    def mark_read(self, request, pk=None):
        notification = self.get_object()
        notification.unread = False
        notification.save(update_fields=["unread"])
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=False, methods=["post"])
    def mark_all_read(self, request):
        Notification.objects.filter(
            recipient=request.user, unread=True
        ).update(unread=False)
        return Response({"status": "ok"})

    @action(detail=False, methods=['get'])
    def unread_count(self, request):
        count = (
            Notification.objects
            .filter(recipient=request.user, unread=True)
            .count()
        )
        return Response({'count': count})


# --- Tags ---
class TagViewSet(viewsets.ModelViewSet):
    queryset = Tag.objects.all().order_by('name')
    serializer_class = TagSerializer

    def get_permissions(self):
        if self.action in ['list', 'retrieve']:
            self.permission_classes = [permissions.IsAuthenticated]
        else:
            self.permission_classes = [IsAdmin]
        return super().get_permissions()
