# models.py در اپلیکیشن 'ideas'

from django.db import models
from django.contrib.auth.models import AbstractUser

# 1. مدل کاربر سفارشی
class User(AbstractUser):
    # 'role' برای سطح دسترسی: 'admin' یا 'user'
    ROLE_CHOICES = (
        ('admin', 'Admin'),
        ('user', 'User'),
    )
    role = models.CharField(max_length=10, choices=ROLE_CHOICES, default='user')
    
    # نیاز به فیلدهای اضافی برای احراز هویت (email, username, password) در AbstractUser پوشش داده شده است.

# 2. مدل دسته‌بندی (برای کمپین‌ها)
class Category(models.Model):
    name = models.CharField(max_length=100, unique=True)
    
    def __str__(self):
        return self.name

# 3. مدل تگ (برای ایده‌ها)
class Tag(models.Model):
    name = models.CharField(max_length=50, unique=True)

    def __str__(self):
        return self.name

# 4. مدل کمپین
class Campaign(models.Model):
    STATUS_CHOICES = (
        ('active', 'Active'),
        ('upcoming', 'Upcoming'),
        ('closed', 'Closed'),
    )
    VISIBILITY_CHOICES = (
        ('open', 'Open to all users'),
        ('invite', 'Invite Only'),
    )

    title = models.CharField(max_length=255)
    description = models.TextField()
    category = models.ForeignKey(Category, on_delete=models.SET_NULL, null=True)
    start_date = models.DateField()
    end_date = models.DateField()
    status = models.CharField(max_length=10, choices=STATUS_CHOICES, default='upcoming')
    visibility = models.CharField(max_length=10, choices=VISIBILITY_CHOICES, default='open')
    
    def __str__(self):
        return self.title

# 5. مدل ایده
class Idea(models.Model):
    STAGE_CHOICES = (
        ('suggest', 'Suggest'),
        ('define', 'Define'),
        ('prioritize', 'Prioritize'),
        ('assign', 'Assign'),
        ('development', 'In Development'),
        ('implemented', 'Implemented'),
        ('on_hold', 'On Hold'),
    )

    title = models.CharField(max_length=255)
    description = models.TextField()
    author = models.ForeignKey(User, on_delete=models.CASCADE, related_name='submitted_ideas')
    campaign = models.ForeignKey(Campaign, on_delete=models.CASCADE, related_name='ideas')
    
    # جزئیات ارزیابی
    utility = models.TextField(blank=True, null=True)
    inventive_step = models.TextField(blank=True, null=True)
    novelty_statement = models.TextField(blank=True, null=True)
    trl = models.CharField(max_length=50, blank=True, null=True) # Technical Readiness Level
    tags = models.ManyToManyField(Tag, blank=True)
    
    # اطلاعات وضعیت و تعاملات
    stage = models.CharField(max_length=20, choices=STAGE_CHOICES, default='suggest')
    likes = models.ManyToManyField(User, related_name='liked_ideas', blank=True)
    dislikes = models.ManyToManyField(User, related_name='disliked_ideas', blank=True)
    score = models.IntegerField(default=0)
    
    # فیلد برای پیگیری آخرین تغییر مرحله
    last_stage_change = models.DateTimeField(auto_now_add=True)
    
    def __str__(self):
        return self.title

# 6. مدل کامنت
class Comment(models.Model):
    text = models.TextField()
    author = models.ForeignKey(User, on_delete=models.CASCADE)
    idea = models.ForeignKey(Idea, on_delete=models.CASCADE, related_name='comments')
    created_at = models.DateTimeField(auto_now_add=True)
    
    def __str__(self):
        return f"Comment by {self.author.username} on {self.idea.title}"

# 7. مدل پیوست (Attachment)
class Attachment(models.Model):
    idea = models.ForeignKey(Idea, on_delete=models.CASCADE, related_name='attachments')
    file = models.FileField(upload_to='attachments/') # برای آپلود فایل‌ها
    file_type = models.CharField(max_length=50, blank=True) # مثلاً 'pdf', 'word', 'excel'
    uploaded_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"Attachment for {self.idea.title}"

# 8. مدل نوتیفیکیشن (Notification) - جدید
class Notification(models.Model):
    # کاربری که نوتیفیکیشن را دریافت می‌کند
    recipient = models.ForeignKey(User, on_delete=models.CASCADE, related_name='notifications')

    # کاربری که باعث ایجاد نوتیفیکیشن شده (مثلاً کسی که کامنت گذاشته)
    actor = models.ForeignKey(User, on_delete=models.CASCADE, related_name='+') # '+' یعنی نیازی به رابطه معکوس نداریم

    # فعل انجام شده (مثلاً "commented on", "liked", "created")
    verb = models.CharField(max_length=255)

    # شیء مورد نظر (مثلاً ایده‌ای که روی آن کامنت گذاشته شده)
    # ما از GenericForeignKey استفاده می‌کنیم تا این فیلد بتواند به هر مدلی اشاره کند (Idea, Campaign, ...)
    from django.contrib.contenttypes.fields import GenericForeignKey
    from django.contrib.contenttypes.models import ContentType

    target_content_type = models.ForeignKey(ContentType, on_delete=models.CASCADE, null=True, blank=True)
    target_object_id = models.PositiveIntegerField(null=True, blank=True)
    target = GenericForeignKey('target_content_type', 'target_object_id')

    # وضعیت خوانده شدن
    unread = models.BooleanField(default=True)

    # زمان ایجاد
    timestamp = models.DateTimeField(auto_now_add=True)

    class Meta:
        ordering = ['-timestamp'] # نوتیفیکیشن‌های جدیدتر بالاتر نمایش داده شوند

    def __str__(self):
        if self.target:
            return f'{self.actor.username} {self.verb} {self.target}'
        return f'{self.actor.username} {self.verb}'